<?php
/**
 * @package Linen
 */

/*
----- Table of Contents

	1.  Load other functions
	2.  Set up theme specific variables
	3.  Define image sizes
	4.  Add custom background
	5.  Add option to limit the content
	6.	Get posts for front page
	7.  Enqueue Client Files
	8.  Print Header Items
				I.    Slider
				II.   Accent Font
				III.  Body Font
	9.  Register Sidebars
	10. Main Menu Fallback
	11. Navigation Function
	12. Get sticky posts count
	13. Featured slider usage
	14. Define theme options
	15. Theme option return functions
				I.    Logo Functions
				II.   Slider Functions
				III.  Subscribe Functions
				IV.   Font Functions
				V.    Layout Functions
				VI.   Footer Functions
	16. Comment form defaults
	17. wp_title override

*/

/*---------------------------------------------------------
	1. Load other functions
------------------------------------------------------------ */
locate_template( array( 'functions' . DIRECTORY_SEPARATOR . 'comments.php' ), true );
locate_template( array( 'functions' . DIRECTORY_SEPARATOR . 'ttf-admin.php' ), true );


if (!class_exists( 'Linen' )) {
	class Linen extends TTFCore {

		/*---------------------------------------------------------
			2. Set up theme specific variables
		------------------------------------------------------------ */
		function Linen () {

			$this->themename = "Linen";
			$this->themeurl = "http://thethemefoundry.com/linen/";
			$this->shortname = "linen";
			$this->domain = "linen";

			add_action( 'init', array(&$this, 'registerMenus' ));
			add_action( 'setup_theme_linen', array(&$this, 'setOptions' ) );

			add_action( 'wp_head', array( &$this, 'printHeaderItems' ) );
			add_action( 'wp_enqueue_scripts', array( &$this, 'enqueueClientFiles' ) );

			add_filter( 'pre_get_posts', array( &$this, 'filter_front_page_posts' ), 10 );

			add_filter( 'comment_form_defaults', array( &$this, 'comment_form_defaults' ) );

			add_filter( 'wp_title', array( &$this, 'page_title' ) );

			$this->add_custom_background();
			$this->update_old_options();

			parent::TTFCore();
		}

		/*---------------------------------------------------------
			3. Define image sizes
		------------------------------------------------------------ */
		function addImageSize() {
			add_image_size( 'featured', 652, 300, true );
			add_image_size( 'index-thumb', 94, 94, true );
			add_image_size( 'single', 620, 9999 );
		}

		/*---------------------------------------------------------
			4. Add custom background
		------------------------------------------------------------ */
		function add_custom_background() {
			if ( function_exists( 'wp_get_theme' ) ) {
				add_theme_support( 'custom-background' );
			} else {
				add_custom_background();
			}
		}

		/**
		 * Updates old options to the new options used in new releases
		 */
		function update_old_options() {
			$new_slider_state = get_option( $this->shortname."_slider_show_on" );

			// If the new slider option has not been set yet, we need to set a value for it
			if ( ! $new_slider_state ) {
				$old_slider_state = get_option( $this->shortname.'_slider' );

				// If the old slider option was turned on, we want the slider on both pages
				if ( $old_slider_state ) {
					update_option( $this->shortname."_slider_show_on", 'front_and_posts' );
				// Otherwise, we don't want it anywhere
				} else {
					update_option( $this->shortname."_slider_show_on", 'none' );
				}
			}
		}

		/*---------------------------------------------------------
			5. Add option to limit the content
		------------------------------------------------------------ */
		function custom_excerpt() {
			global $post;
			$content = '';

			if ( $post->post_excerpt ) {
				// We allow certain tags in the excerpt because people may be expecting HTML to be allowed here
				$content =
					wp_kses( $post->post_excerpt, array(
						'a' => array( 'href' => array(), 'title' => array() ),
						'br' => array(),
						'em' => array(),
						'strong' => array() ) );
				$content = wpautop( $content );
				$content .= '<p><a class="more-link" href="'. get_permalink() .'">' . __( 'Read more', 'linen' ) . '</a></p>';
			}

			return $content;
		}

		function custom_content($limit) {
			$link = get_permalink();
			$read_more_link = '<a class="more-link" href="'. $link .'">' . __( 'Read more', 'linen' ) . '</a>';

			// All HTML is stripped from the content
			$content = explode( ' ', strip_tags( get_the_content('...') ) );

			// Only truncate content if longer than the specified limit
			if ( count($content) > $limit ) {
				array_splice( $content, $limit );
				$content = implode( " ", $content ) . '...' . $read_more_link;
			} else {
				$content = implode( " ", $content ) . $read_more_link;
			}

			$content = preg_replace( '/<img[^>]+./','', $content );
			$content = preg_replace( '|\[(.+?)\](.+?\[/\\1\])?|s','', $content );
			$content = wpautop( $content );
			$content = str_replace( ']]>', ']]&gt;', $content );
			return $content;
		}

		/*---------------------------------------------------------
			6. Get front page posts
		------------------------------------------------------------ */

		function filter_front_page_posts( $query ) {
			global $wp_query;

			// Apply this filter as long as the slider is enabled and we're on the front or posts page
			$apply_filter = $this->slider_in_use() && ( is_front_page() || $wp_query->is_posts_page );

			if ( $apply_filter ) {
				if ( $this->sliderIncludeStickies() ) {
					$query->set( 'ignore_sticky_posts', 1 );
				} else {
					$query->set( 'post__not_in', get_option( 'sticky_posts' ) );
				}
			}

			return $query;
		}

		/*---------------------------------------------------------
			7. Enqueue Client Files
		------------------------------------------------------------ */
		function enqueueClientFiles() {
			global $wp_styles;

			if ( ! is_admin() ) {
				wp_enqueue_style(
					'linen-style',
					get_bloginfo( 'stylesheet_url' ),
					'',
					null
				);

				wp_enqueue_style(
					'linen-ie-style',
					get_template_directory_uri() . '/stylesheets/ie.css',
					array( 'linen-style' ),
					null
				);
				$wp_styles->add_data( 'linen-ie-style', 'conditional', 'lt IE 8' );

				$protocol = is_ssl() ? 'https' : 'http';

				if ( ( 'disable' != $this->accentFont() ) ) {
					wp_enqueue_style(
						'linen-accent-font-style',
						"$protocol://fonts.googleapis.com/css?family=" . $this->accentFont(),
						array( 'linen-style' ),
						null
					);
				}

				if ( ( 'disable' != $this->bodyFont() ) && ( $this->bodyFont() != $this->accentFont() ) ) {
					wp_enqueue_style(
						'linen-body-font-style',
						"$protocol://fonts.googleapis.com/css?family=" . $this->bodyFont(),
						array( 'linen-style' ),
						null
					);
				}

				if ( is_singular() ) {
					wp_enqueue_script( 'comment-reply' );
				}

				wp_enqueue_script( 'jquery' );

				wp_enqueue_script(
					'linen',
					get_template_directory_uri() . '/javascripts/linen.js',
					array( 'jquery' ),
					null
				);
			}
		}

		/*---------------------------------------------------------
			8. Print Header Items
		------------------------------------------------------------ */
		function printHeaderItems() {
			global $wp_query;
			/*---------------------------------------------------------
				I. Slider
			------------------------------------------------------------ */
			if ( ( is_front_page() || $wp_query->is_posts_page ) && $this->use_javascript_slider() ) : ?>
				<?php
					$slidespeed = 350;
					if ($this->sliderSpeed() != '' )
						$slidespeed = $this->sliderSpeed();
					if ($this->sliderStart() == 'true' )
						$play = $this->sliderDelay();
					else
						$play = 0;
				?>
				<script>
					jQuery(function(){
						jQuery("#featured").slides({
							generatePagination: false,
							play: <?php echo esc_js( $play ); ?>,
							slideSpeed: <?php echo esc_js( $slidespeed ); ?>
						});
						jQuery('#slides a.prev, #slides a.next').show();
					});
					document.write("<style type='text/css'>#slides div.show-slide { display: none; }</style>");
				</script>
				<?php
			endif;

			/*---------------------------------------------------------
				II. Accent Font
			------------------------------------------------------------ */
			if ( $this->accentFont() == 'Cabin:bold' )
				$accent_font = 'Cabin';
			elseif ( $this->accentFont() == 'Droid+Sans' )
				$accent_font = 'Droid Sans';
			elseif ( $this->accentFont() == 'Droid+Serif' )
				$accent_font = 'Droid Serif';
			else
				$accent_font = $this->accentFont();

			if ( 'disable' != $this->accentFont() ) : ?>
				<style type="text/css">#title, .nav a, .slide-content h2, .slide-content a.more-link, .entry h2.title, .single-post-title, .post-date, .entry a.more-link, div.post-comments a, .entry h2, .entry h3, .entry h4, .post-footer, .navigation, h1.pagetitle, h2.pagetitle, .entries li span, #sidebar h2.widgettitle, .comment-number, div.c-head, div.reply, div.cancel-comment-reply, h3#reply-title, .footer-column h2, #copyright { font-family: '<?php echo $accent_font ?>', Helvetica, Arial, sans-serif; }</style>
				<?php
			endif;

			/*---------------------------------------------------------
				III. Body Font
			------------------------------------------------------------ */
			if ( $this->bodyFont() == 'Droid+Sans' )
				$body_font = 'Droid Sans';
			elseif ( $this->bodyFont() == 'Droid+Serif' )
				$body_font = 'Droid Serif';
			elseif ( $this->bodyFont() == 'Cantarell:regular,italic' )
				$body_font = 'Cantarell';
			elseif ( $this->bodyFont() == 'Nobile:regular,italic' )
				$body_font = 'Nobile';
			elseif ( $this->bodyFont() == 'Puritan:regular,italic' )
				$body_font = 'Puritan';
			else
				$body_font = $this->bodyFont();

			if ( 'disable' != $this->bodyFont() ) : ?>
				<style type="text/css">body { font-family: '<?php echo $body_font ?>', Georgia, Times, serif; }</style>
				<?php
			endif;

		}

		/*---------------------------------------------------------
			9. Register Sidebars
		------------------------------------------------------------ */
		function registerSidebars() {

			register_sidebar( array(
				'name'=> __( 'Sidebar', 'linen' ),
				'id' => 'sidebar_1',
				'before_widget' => '<li id="%1$s" class="widget %2$s">',
				'after_widget' => '</li>',
				'before_title' => '<h2 class="widgettitle">',
				'after_title' => '</h2>',
			) );
			register_sidebar( array(
				'name'=> __( 'Footer 1', 'linen' ),
				'id' => 'footer_1',
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget' => '</div>',
				'before_title' => '<h2 class="widgettitle">',
				'after_title' => '</h2>',
			) );
			register_sidebar( array(
				'name'=> __( 'Footer 2', 'linen' ),
				'id' => 'footer_2',
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget' => '</div>',
				'before_title' => '<h2 class="widgettitle">',
				'after_title' => '</h2>',
			) );
			register_sidebar( array(
				'name'=> __( 'Footer 3', 'linen' ),
				'id' => 'footer_3',
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget' => '</div>',
				'before_title' => '<h2 class="widgettitle">',
				'after_title' => '</h2>',
			) );
			register_sidebar( array(
				'name'=> __( 'Footer 4', 'linen' ),
				'id' => 'footer_4',
				'before_widget' => '<div id="%1$s" class="widget %2$s">',
				'after_widget' => '</div>',
				'before_title' => '<h2 class="widgettitle">',
				'after_title' => '</h2>',
			) );

		}

		/*---------------------------------------------------------
			10. Main Menu Fallback
		------------------------------------------------------------ */
		function main_menu_fallback() {
			?>
			<div id="navigation" class="clear">
				<ul class="nav">
					<?php
						wp_list_pages( 'title_li=' );
					?>
				</ul>
			</div>
			<?php
			}

		/*---------------------------------------------------------
			11. Navigation Functions
		------------------------------------------------------------ */
		function registerMenus() {
			register_nav_menu( 'nav-1', __( 'Top Navigation' ) );
		}

		/*---------------------------------------------------------
			12. Get sticky posts count
		------------------------------------------------------------ */
		function get_sticky_posts_count() {
			global $wpdb;
			$sticky_posts = array_map( 'absint', (array) get_option('sticky_posts') );
			return count($sticky_posts) > 0 ? $wpdb->get_var( "SELECT COUNT( 1 ) FROM $wpdb->posts WHERE post_type = 'post' AND post_status = 'publish' AND ID IN (".implode(',', array_map( 'intval', $sticky_posts ) ).")" ) : 0;
		}

		/*---------------------------------------------------------
			13. Featured slider usage
		------------------------------------------------------------ */
		function use_featured_header() {
			// We don't want the header if the slider isn't in use or we're beyond page 1
			if ( ! $this->slider_in_use() || is_paged() ) {
				return false;
			}

			$show_slider_on = $this->slider_state();

			// Now we check the option and base the header display on which page we're on
			// Note that is_home() returns true on the posts page
			switch ( $show_slider_on ) {
				case 'front_and_posts':
					return ( is_front_page() || is_home() );
				case 'front_only':
					return is_front_page();
				case 'posts_only':
					return is_home();
				default:
					return false;
			}
		}

		/**
		 * We can use the slider if it is not completely turned off and there are sticky posts
		 */
		function slider_in_use() {
			$show_slider_on = $this->slider_state();

			return ( 'none' != $show_slider_on ) && ( $this->get_sticky_posts_count() >= 1 );
		}

		function use_javascript_slider() {
			return $this->use_featured_header() && $this->get_sticky_posts_count() >= 2;
		}

		/*---------------------------------------------------------
			14. Define theme options
		------------------------------------------------------------ */
		function setOptions() {

			/*
				OPTION TYPES:
				- checkbox: name, id, desc, std, type
				- radio: name, id, desc, std, type, options
				- text: name, id, desc, std, type
				- colorpicker: name, id, desc, std, type
				- select: name, id, desc, std, type, options
				- textarea: name, id, desc, std, type, options
			*/

			$this->options = array(

				array(
					"name" => __( 'Custom logo image', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Enable custom logo image', 'linen' ),
					"id" => $this->shortname."_logo",
					"desc" => __( 'Check to use a custom logo in the header.', 'linen' ),
					"std" => "false",
					"type" => "checkbox"),

				array(
					"name" => __( 'Logo URL', 'linen' ),
					"id" => $this->shortname."_logo_img",
					"desc" => __( 'Upload an image or enter an URL for your image.', 'linen' ),
					"std" => '',
					"upload" => true,
					"class" => "logo-image-input",
					"type" => "upload"),

				array(
					"name" => __( 'Logo image <code>&lt;alt&gt;</code> tag', 'linen' ),
					"id" => $this->shortname."_logo_img_alt",
					"desc" => __( 'Specify the <code>&lt;alt&gt;</code> tag for your logo image.', 'linen' ),
					"std" => '',
					"type" => "text"),

				array(
					"name" => __( 'Display tagline', 'linen' ),
					"id" => $this->shortname."_tagline",
					"desc" => __( 'Check to show your tagline below your logo.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Featured slider', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Show featured slider', 'linen' ),
					"desc" => __( 'Choose where you want the featured slider to display. By default, the slider is on both the front and posts page. If you have <strong>not</strong> set a static front page on the <em>Settings &rarr; Reading</em> screen, your front page is the same as your posts page.', 'linen' ),
					"id" => $this->shortname."_slider_show_on",
					"std" => 'front_and_posts',
					"type" => "select",
					"options" => array(
						"front_and_posts" => __( 'On front page and posts page', 'linen' ),
						"front_only" => __( 'On front page only', 'linen' ),
						"posts_only" => __( 'On posts page only', 'linen' ),
						"none" => __( 'Nowhere', 'linen' ) ) ),

				array(
					"name" => __( 'Autostart', 'linen' ),
					"id" => $this->shortname."_slider_start",
					"desc" => __( 'Check to start your featured slider automatically.', 'linen' ),
					"std" => "false",
					"type" => "checkbox"),

				array(
					"name" => __( 'Autostart delay', 'linen' ),
					"id" => $this->shortname."_slider_delay",
					"desc" => __( 'Delay before the autostart and the delay between slides in milliseconds (1000 = 1 second).', 'linen' ),
					"std" => "4000",
					"type" => "text"),

				array(
					"name" => __( 'Slide animation speed', 'linen' ),
					"id" => $this->shortname."_slider_speed",
					"desc" => __( 'Speed of the sliding animation in milliseconds (1000 = 1 second).', 'linen' ),
					"std" => "350",
					"type" => "text"),

				array(
					"name" => __( 'Don\'t hide sticky posts', 'linen' ),
					"id" => $this->shortname."_slider_include_stickies",
					"desc" => __( 'Check to show sticky posts in the standard post list in addition to the featured slider. By default, sticky posts show only in the featured slider.', 'linen' ),
					"std" => "false",
					"type" => "checkbox"),

				array(
					"name" => __( 'Subscribe links', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Enable Twitter', 'linen' ),
					"id" => $this->shortname."_twitter_toggle",
					"desc" => __( 'Hip to Twitter? Check this box. Please set your Twitter username in the Twitter menu.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Enable Facebook', 'linen' ),
					"id" => $this->shortname."_facebook_toggle",
					"desc" => __( 'Check this box to show a link to your Facebook page.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Enable Flickr', 'linen' ),
					"id" => $this->shortname."_flickr_toggle",
					"desc" => __( 'Check this box to show a link to Flickr.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Enable Google+', 'linen' ),
					"id" => $this->shortname."_google_plus_toggle",
					"desc" => __( 'Check this box to show a link to Google+.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Disable all', 'linen' ),
					"id" => $this->shortname."_follow_disable",
					"desc" => __( 'Check this box to hide all follow icons (including RSS). This option overrides any other settings.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Twitter name', 'linen' ),
					"id" => $this->shortname."_twitter",
					"desc" => __( 'Enter your Twitter name.', 'linen' ),
					"std" => '',
					"type" => "text"),

				array(
					"name" => __( 'Facebook link', 'linen' ),
					"id" => $this->shortname."_facebook",
					"desc" => __( 'Enter your Facebook link.', 'linen' ),
					"std" => '',
					"type" => "text"),

				array(
					"name" => __( 'Flickr link', 'linen' ),
					"id" => $this->shortname."_flickr",
					"desc" => __( 'Enter your Flickr link.', 'linen' ),
					"std" => '',
					"type" => "text"),

				array(
					"name" => __( 'Google+ link', 'linen' ),
					"id" => $this->shortname."_google_plus",
					"desc" => __( 'Enter your Google+ link.', 'linen' ),
					"std" => '',
					"type" => "text"),

				array(
					"name" => __( 'Typography', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Accent font', 'linen' ),
					"desc" => __( 'Used for accents and headers. Fallback font stack is "Helvetica, Arial, sans-serif". Added page weight is in parentheses.' ),
					"id" => $this->shortname."_accent_font",
					"std" => '',
					"type" => "select",
					"options" => array(
						"Arvo" => __( 'Arvo (27kb)', 'linen' ),
						"Cabin:bold" => __( 'Cabin (39kb)', 'linen' ),
						"Copse" => __( 'Copse (22kb)', 'linen' ),
						"Droid+Sans" => __( 'Droid Sans (26kb)', 'linen' ),
						"Droid+Serif" => __( 'Droid Serif (28kb)', 'linen' ),
						"disable" => __( 'Helvetica', 'linen' ),
						"Lato" => __( 'Lato (49kb)', 'linen' ) ) ),

				array(
					"name" => __( 'Body font', 'linen' ),
					"desc" => __( 'Used for the body text. Fallback font stack is "Georgia, Times, serif". Added page weight is in parentheses.' ),
					"id" => $this->shortname."_body_font",
					"std" => 'disable',
					"type" => "select",
					"options" => array(
						"Cantarell:regular,italic" => __( 'Cantarell (53kb)', 'linen' ),
						"Droid+Sans" => __( 'Droid Sans (26kb)', 'linen' ),
						"Droid+Serif" => __( 'Droid Serif (28kb)', 'linen' ),
						"disable" => __( 'Georgia', 'linen' ),
						"Lato" => __( 'Lato (49kb)', 'linen' ),
						"Neuton" => __( 'Neuton (18kb)', 'linen' ),
						"Nobile:regular,italic" => __( 'Nobile (57kb)', 'linen' ),
						"Puritan:regular,italic" => __( 'Puritan (60kb)', 'linen' ) ) ),

				array(
					"name" => __( 'Layout', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Disable Sidebar', 'linen' ),
					"id" => $this->shortname."_sidebar_disable",
					"desc" => __( 'Check this box to disable the sidebar. This removes the sidebar from <strong>all</strong> pages and posts.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Disable Footer', 'linen' ),
					"id" => $this->shortname."_footer_disable",
					"desc" => __( 'Check this box to disable the footer widget area. This removes the footer from <strong>all</strong> pages and posts.', 'linen' ),
					"std" => '',
					"type" => "checkbox"),

				array(
					"name" => __( 'Footer', 'linen' ),
					"type" => "subhead"),

				array(
					"name" => __( 'Copyright notice', 'linen' ),
					"id" => $this->shortname."_copyright_name",
					"desc" => __( 'Your name or the name of your business.', 'linen' ),
					"std" => '',
					"type" => "text")
			);
		}

		/*---------------------------------------------------------
			15. Theme option return functions
		------------------------------------------------------------ */

			/*---------------------------------------------------------
				I. Logo Functions
			------------------------------------------------------------ */
			function logoState () {
				return get_option($this->shortname.'_logo' );
			}
			function logoName () {
				return get_option( $this->shortname.'_logo_img' );
			}
			function logoAlt () {
				return get_option($this->shortname.'_logo_img_alt' );
			}
			function logoTagline () {
				return get_option($this->shortname.'_tagline' );
				}

			/*---------------------------------------------------------
				II. Slider Functions
			------------------------------------------------------------ */
			function slider_state () {
				return get_option($this->shortname.'_slider_show_on' );
			}
			function sliderStart () {
				return get_option($this->shortname.'_slider_start' );
			}
			function sliderDelay () {
				return get_option($this->shortname.'_slider_delay' );
			}
			function sliderSpeed () {
				return get_option($this->shortname.'_slider_speed' );
			}
			function sliderIncludeStickies () {
				return get_option($this->shortname.'_slider_include_stickies' );
			}

			/*---------------------------------------------------------
				III. Subscribe Functions
			------------------------------------------------------------ */
			function twitter() {
				return stripslashes( wp_filter_post_kses(get_option($this->shortname.'_twitter' )) );
			}
			function twitterToggle() {
				return get_option($this->shortname.'_twitter_toggle' );
			}
			function facebook() {
				return stripslashes( wp_filter_post_kses(get_option($this->shortname.'_facebook' )) );
			}
			function facebookToggle() {
				return get_option($this->shortname.'_facebook_toggle' );
			}
			function flickr() {
				return wp_filter_post_kses(get_option($this->shortname.'_flickr' ));
			}
			function flickrToggle() {
				return get_option($this->shortname.'_flickr_toggle' );
			}
			function googlePlus() {
				return get_option( $this->shortname."_google_plus" );
			}
			function googlePlusToggle() {
				return get_option( $this->shortname."_google_plus_toggle" );
			}
			function followDisable() {
				return get_option($this->shortname.'_follow_disable' );
			}

			/*---------------------------------------------------------
				IV. Font Functions
			------------------------------------------------------------ */
			function accentFont () {
				return get_option($this->shortname.'_accent_font', 'Arvo' );
			}
			function bodyFont () {
				return get_option($this->shortname.'_body_font', 'disable' );
			}

			/*---------------------------------------------------------
				V. Layout
			------------------------------------------------------------ */
			function sidebarDisable() {
				return get_option($this->shortname.'_sidebar_disable' );
			}
			function footerDisable() {
				return get_option($this->shortname.'_footer_disable' );
			}

			/*---------------------------------------------------------
				VI. Footer Functions
			------------------------------------------------------------ */
			function copyrightName() {
				return stripslashes( wp_filter_post_kses(get_option($this->shortname.'_copyright_name' )) );
			}

		/*---------------------------------------------------------
		    16. Comment form defaults
		------------------------------------------------------------ */
		function comment_form_defaults( $defaults ) {
			// Get author information
			$comment_author       = ( isset( $_POST['author'] ) ) ? trim( strip_tags( $_POST['author'] ) ) : null;
			$comment_author_email = ( isset( $_POST['email'] ) )  ? trim( $_POST['email'] ) : null;
			$comment_author_url   = ( isset( $_POST['url'] ) )    ? trim( $_POST['url'] ) : null;

			// Get current user
			$user  = wp_get_current_user();
			$login = $user->user_login;

			// Set up default field
			$req   = get_option( 'require_name_email' );
			$field = '<p><label for="%1$s" class="comment-field">%2$s</label><input class="text-input" type="text" name="%1$s" id="%1$s" value="%3$s" size="22" /></p>';

			// Set defaults
			$defaults['comment_field']        = '<fieldset><label for="comment" class="comment-field"><small>' . _x( 'Comment', 'noun', 'linen' ) . '</small></label><textarea id="comment" name="comment" cols="50" rows="10" aria-required="true"></textarea></fieldset>';
			$defaults['comment_notes_before'] = '';
			$defaults['comment_notes_after']  = sprintf(
				'<p class="guidelines">%3$s</p>' . "\n" . '<p class="comments-rss"><a href="%1$s">%2$s</a></p>',
				esc_attr( get_post_comments_feed_link() ),
				__( 'Subscribe to this comment feed via RSS', 'linen' ),
				__( 'You may use basic HTML in your comments. Your email address will not be published.', 'linen' )
			);
			$defaults['fields']               = array(
				'author' => sprintf(
					$field,
					'author',
					$req ? __( 'Name <span>(required)</span>:', 'linen' ) : __( 'Name:', 'linen' ),
					esc_attr( $comment_author )
				),
				'email'  => sprintf(
					$field,
					'email',
					$req ? __( 'Email <span>(required)</span>:', 'linen' ) : __( 'Email:', 'linen' ),
					esc_attr( $comment_author_email )
				),
				'url'    => sprintf(
					$field,
					'url',
					__( 'Website:', 'linen' ),
					esc_attr( $comment_author_url )
				),
			);
			$defaults['label_submit']         = __( 'Submit Comment', 'linen' );
			$defaults['logged_in_as']         = '<p class="com-logged-in">' . sprintf( __( 'Logged in as <a href="%1$s">%2$s</a>. <a href="%3$s" title="Log out of this account">Log out &raquo;</a>', 'linen' ), admin_url( 'profile.php' ), $login, wp_logout_url( apply_filters( 'the_permalink', get_permalink() ) ) ) . '</p>';
			$defaults['title_reply']          = __( 'Leave a Reply', 'linen' );
			$defaults['title_reply_to']       = __( 'Leave a comment to %s', 'linen' );

			return $defaults;
		}

		/*---------------------------------------------------------
		    17. wp_title override
		------------------------------------------------------------ */
		function page_title( $title ) {
			// We don't want to affect RSS feeds
			if ( is_feed() ) {
				return;
			}

			if ( is_front_page() ) {
				return get_bloginfo( 'name' );
			} elseif ( is_404() ) {
				return __( 'Page Not Found | ', 'linen' ) . get_bloginfo( 'name' );
			} elseif ( is_search() ) {
				return sprintf( __( "Search results for '%s' | ", 'linen' ), get_search_query() ) . get_bloginfo( 'name' );
			} else {
				return trim( $title ) . ' | ' . get_bloginfo( 'name' );
			}
		}

	}
}

/* SETTING EVERYTHING IN MOTION */
function load_linen_pro_theme() {
	$GLOBALS['linen'] = new Linen;
}

add_action( 'after_setup_theme', 'load_linen_pro_theme' );